%[text] # Two-Hour Simulation
%%
%[text] **Description:** Let's look at a simulation of user behavior over two hours instead of 10 minutes. The computer will generate a scenario based on a random positive whole number that you enter.
seed = 0; %[control:editfield:49bd]{"position":[8,9]}
  %[control:button:84bf]{"position":[1,2]}

if seed < 0
    error("Please enter a number that is not negative.")
elseif seed > 1000000
    error("Please enter a number that is smaller than 1,000,000")
elseif seed ~= round(seed)
    error("You should enter a whole number.")
end

rng(seed);

% Angles = CreateRandomScenario(1200); % Random scenario of 120 minutes
TNData = CreateRandomScenario(1200);
%%
  %[control:button:8f8c]{"position":[1,2]}
GenerateSummaryPlot(TNData)
PrintSummaryStatistics(TNData)
%%
function TextNeckPlots = SetUpPlots(TotTimeSteps)
% SETUPPLOTS Function to set up 3D plots for torso and head, and a 2D plot for text neck over time
%
% Output Arguments:
%     TextNeckPlots - structure containing handles for the plots
%         .CurrPlot - handle for the current plot of text neck over time
%         .SensorPlot - handle for the sensor plot
%         .TextNeckAxes - axes for the text neck plot
%         .head - handle for the head plot
%         .torso - handle for the torso plot

clf % Clear current figure
f = figure('Position', [0 0 1100 400]); % Create a new figure with specified position
t = tiledlayout(1,2); % Create a tiled layout for multiple plots
nexttile; % Move to the next tile in the layout
% Plot torso as a 3D line
torso = plot3([4 4], [4 4], [4 8], 'SeriesIndex', 1, 'LineWidth', 3);
hold on % Hold the current plot to overlay additional plots
% Plot arms as a 3D line
arms = plot3([3 2.5 3 4 5 5.5 5], zeros(1, 7) + 4, [5 6 7 7 7 6 5], 'SeriesIndex', 1, 'LineWidth', 3);
% Plot legs as a 3D line
legs = plot3([3 3 3 4 5 5 5], [3.5, zeros(1, 5) + 4, 3.5], [1 1 3 4 3 1 1], 'SeriesIndex', 1, 'LineWidth', 3);
theta = linspace(0, 2*pi, 1000); % Create an array of angles for the head
x = 0.5 * cos(theta) + 4; % X coordinates for the head
y = zeros(1, length(theta)) + 4; % Y coordinates for the head
z = sin(theta) + 9; % Z coordinates for the head
% Plot head as a 3D line
head = plot3(x, y, z, 'SeriesIndex', 1, 'LineWidth', 3);
MyColors = colororder; % Get the current color order
% Fill areas with transparency to represent different sections
fill3([0 0 10 10], [0 10 10 0], [1 1 1 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([10 10 10 10], [0 10 10 0], [1 1 10 10], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
fill3([0 0 10 10], [10 10 10 10], [1 10 10 1], MyColors(1,:), 'FaceAlpha', 0.1, 'EdgeColor', MyColors(1,:));
xlim([0 10]) % Set limits for the x-axis
ylim([0 10]) % Set limits for the y-axis
zlim([0 11]) % Set limits for the z-axis
hold off
ax = gca; % Get current axes
ax.Visible = 'off'; % Hide the axes
SensorPlot = PlotSensor(torso); % Call function to plot sensor data

% Set up the axes for the text neck plot
TextNeckAxes = axes(t);
TextNeckAxes.Layout.Tile = 2;   % Position the axes in the second tile
CurrPlot = plot(0, false, 'b'); % Initialize the current plot
xlim([0 TotTimeSteps]);         % Set limits of the text neck plot in 1/100 minute increments
ylim([-0.25 1.25]);             % Set limits of the text neck plot
xlabel('Minutes');              % Label for the x-axis
ylabel('Text Neck?');           % Label for the y-axis
title('Text Neck Over Time');   % Title for the plot
yticks([0 1]);                  % Set y-ticks
xticks(0:100:TotTimeSteps);     % Set x-ticks
xticklabels(0:10:TotTimeSteps/10); % Set x-tick labels
xtickangle(0);                  % Set angle for x-tick labels
yticklabels({'No', 'Yes'});     % Set y-tick labels

% Create output structure
TextNeckPlots.CurrPlot = CurrPlot;
TextNeckPlots.SensorPlot = SensorPlot;
TextNeckPlots.TextNeckAxes = TextNeckAxes;
TextNeckPlots.head = head;
TextNeckPlots.torso = torso;
end

function TNData = InitializeTextNeckData(TotTimeSteps)
TNData.TextNeck = false;
TNData.History = zeros([1 TotTimeSteps]);
end

function SensorPlot = PlotSensor(torso)
hold on
SensorPlot = scatter3(torso.XData(2), torso.YData(2), torso.ZData(2), 'SizeData', 30, 'SeriesIndex', 'none');
hold off

ax = gca;
ax.Visible = 'off';
end

function TNData = CreateRandomScenario(TotTimeSteps)
TNPlot = SetUpPlots(TotTimeSteps); % With step size 0.1 min
TNData = InitializeTextNeckData(TotTimeSteps);
%AngleHistory = zeros([1 TotTimeSteps]);
direction = randi(2);
speed = 0;

for timeStep = 1:1200 % in 0.1 minute increments
    if ShouldChangeDirection(timeStep)
        direction = randi(2);
        speed = GenerateRandomSpeed();
    end
    % Move the head down or up based on the mode
    [TNPlot,TNData] = CheckPosition(timeStep, TNPlot, TNData);
    ApplyHeadRotation(TNPlot, direction, speed, TNData.CurAngle)
%    AngleHistory(timeStep) = TNData.CurAngle;
end
end

function [TextNeck,angle] = DetermineNeckPosition(TNPlot,TNAngle)
arguments
    TNPlot 
    TNAngle (1,1) double {mustBeInteger,mustBePositive} = 20 
end

angle = CalculateHeadAngle(TNPlot);

% Compute TextNeck values for the current position
if abs(angle) < (180-TNAngle)
    % Set TextNeck=true if the neck is bent more than 20 degrees
    TextNeck = true;
    % If the neck angle moves within 20 degrees of 180
elseif abs(angle) >= (180-TNAngle)
    TextNeck = false;
end
end

function [TNPlot,TNData] = CheckPosition(i, TNPlot, TNData)
% CHECKPOSITION Function to check the position of the neck
%
% Input Arguments:
%     i      - Current timestep index (0.01 minutes each)
%     TNPlot - Structure containing 
%        CurrPlot - a line plot of the textneck status out/0 vs in/1 
%        SensorPlot - a scatter showing where the sensor is located (at the
%              head/neck point on the figure)
%        TextNeckAxes - a handle to the axes containing CurrPlot
%        head - a handle to the line plot of the figure's head
%        torso - a handle to the line plot of the figure's torso
%     TNData - Structure containing data related to neck position
%        ContiguousTimeInTNP - the time in TextNeck without leaving
%        TextNeck - the current state, true is in TextNeck position
%        History - an in/out record for TextNeck position
%
% Output Arguments:
%     TNPlot - Updated plot data structure
%     TNData - Updated data structure with TextNeck History

% Update TNData value at time i

% First, calculate whether we are in TextNeck at all
[TextNeck,TNData.CurAngle] = DetermineNeckPosition(TNPlot);

TNData.History(i) = double(TextNeck);

TNPlot.CurrPlot.XData = [TNPlot.CurrPlot.XData i];
TNPlot.CurrPlot.YData = [TNPlot.CurrPlot.YData TNData.History(i)];

% % Pause for a short duration every 0.2 minutes
% if mod(i,10) == 0
%     pause(0.01)
% end
% Ensure the plot is updated with the latest data
if mod(i,10) == 0
    pause(0.05);
end
end

function GenerateSummaryPlot(TNData)
% GENERATESUMMARYPLOT Create final summary plot of text neck over time
%
% Input Arguments:
%   numTimeSteps - Total number of time steps simulated
%   AllPlots - Array of plot handles to copy into summary figure

% Create new figure with wide aspect ratio for time series
f = figure('Position', [0 0 5000 1000]);
a = axes(f);
numTimeSteps = numel(TNData.History);

% Plot TextNeck History
plot(a,TNData.History)

% Configure axes
xlim([0 numTimeSteps]);
ylim([-0.25 1.25]);
xlabel('Minutes');
ylabel('Text Neck?');
title('Text Neck Over Time');

% Set up ticks (assuming 10 time steps = 1 minute)
yticks([0 1]);
yticklabels({'No', 'Yes'});
xticks(0:100:numTimeSteps);
xticklabels(0:10:(numTimeSteps/10));
xtickangle(0);
end


function PrintSummaryStatistics(TNData)
% PRINTSUMMARYSTATISTICS Display summary statistics to console
%
% Input Arguments:
%   TNData.history contains timestamped logical in/out of text neck history
%
% Notes:
%   Assumes 10 time steps = 1 minute for time conversion

totalMinutes = sum(TNData.History) / 10;
totalEntries = sum(diff(TNData.History) == 1);
fprintf('Number of times entering text neck position: %i\n', totalEntries);
fprintf('Total time in text neck position: %.1f minutes\n', totalMinutes);
end

function speed = GenerateRandomSpeed()
% GENERATERANDOMSPEED Generate random rotation speed between 1 and 3 degrees/step
    speed = 1 + (3 - 1) * rand;
end


function shouldChange = ShouldChangeDirection(timeStep)
% SHOULDCHANGEDIRECTION Determine if direction should change at current time step
%
% Direction changes every 10 time steps by checking if the tens digit changes
    shouldChange = mod(floor(timeStep/10), 2) ~= mod(floor((timeStep-1)/10), 2);
end


function angle = CalculateHeadAngle(TNPlot)
% CALCULATEHEADANGLE Calculate angle between head and torso orientation vectors
%
% Output Arguments:
%   Vec1 - Vector from torso reference point to head position
%   Vec2 - Vector representing torso orientation (from point 1 to point 2)
%   angle - Angle between Vec1 and Vec2 in degrees

head = TNPlot.head;
torso = TNPlot.torso;
    % Extract key points from head and torso
    headPoint = [head.XData(250), head.YData(250), head.ZData(250)];
    torsoReferencePoint = [torso.XData(2), torso.YData(2), torso.ZData(2)];
    torsoTopPoint = [torso.XData(1), torso.YData(1), torso.ZData(1)];
    
    % Calculate vectors
    Vec1 = headPoint - torsoReferencePoint;
    Vec2 = torsoTopPoint - torsoReferencePoint;
    
    % Calculate angle using vector cross product and dot product
    angle = atan2d(norm(cross(Vec1, Vec2)), dot(Vec1, Vec2));
end


function ApplyHeadRotation(TNPlot, direction, speed, currentAngle)
% APPLYHEADROTATION Rotate head based on direction with angle constraints
%
% Input Arguments:
%   head - Handle to head graphics object
%   torso - Handle to torso graphics object
%   direction - Rotation direction (1 = down/forward, 2 = up/back)
%   speed - Rotation speed in degrees
%   currentAngle - Current angle between head and torso in degrees
%
% Notes:
%   - Direction 1 (down): Only rotate if angle >= 120° (prevents over-rotation)
%   - Direction 2 (up): Only rotate if angle <= 177° (prevents hyper-extension)
head = TNPlot.head;
torso = TNPlot.torso;

    % Get rotation pivot point (torso reference point)
    pivotPoint = [torso.XData(2), torso.YData(2), torso.ZData(2)];
    
    if direction == 1
        % Rotate head forward/down (around positive x-axis)
        if currentAngle >= 120
            rotate(head, [1 0 0], speed, pivotPoint);
        end
    elseif direction == 2
        % Rotate head backward/up (around negative x-axis)
        if currentAngle <= 177
            rotate(head, [-1 0 0], speed, pivotPoint);
        end
    end
end

%[appendix]{"version":"1.0"}
%---
%[metadata:view]
%   data: {"layout":"hidecode","rightPanelPercent":40}
%---
%[control:editfield:49bd]
%   data: {"defaultValue":0,"label":"Enter a whole number: ","run":"Nothing","valueType":"Double"}
%---
%[control:button:84bf]
%   data: {"label":"Run simulation","run":"Section"}
%---
%[control:button:8f8c]
%   data: {"label":"View plot and stats","run":"Section"}
%---
